/**
*
* NEKJE - Nur ein kleiner Java-Editor
*/

import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.awt.event.*;
import java.awt.*;
import java.awt.print.*;
import java.io.*;
import java.util.*;
import java.net.*;
import java.text.*;

/**
* Klasse Nekje
* 
* Hauptklasse des Editors
*/
public class Nekje extends JFrame {
	/**
	* Referenz auf das Menue-Panel
	*/
	protected JMenuBar nekjeMenuBar;
	
	/**
	* Referenz auf Toolbar der Anwendung
	*/
	protected JToolBar nekjeToolBar;
	
	/**
	* Referenz auf das Workingbench-Panel
	*/
	protected NekjeWBPanel wbPanel;
	
	/**
	* Referenz auf Ausgabeconsole
	*/
	protected NekjeConsole nekjeConsole;
	
	/**
	* Referenz auf Programm-Properties
	*/
	protected static Properties props;
		
	/**
	*	Referenz auf Programm-Icon	
	*/
	protected ImageIcon nekjeIcon;
	
	/**
	*	Referenz auf Quelltexte	
	*/

	Vector codeSnippets;
	
	boolean indentPlugin = false;
	
	protected NekjeReference referenceBrowser;
	
	
	
	final static String version = "Nekje Version 00-04-15";
	
	/**
	* Konstruktor fuer die Klasse Nekje
	*/
	Nekje( String properties ) {
		super( "Nekje" );
		
		SplashScreen sscreen = new SplashScreen( "images/nekje.gif" );
		sscreen.setVisible(true);
		
		sscreen.setText( version, 20 );
		
		nekjeIcon = new ImageIcon(getClass().getResource("images/nekje_icon.gif"));
		setIconImage( nekjeIcon.getImage() );
		
		sscreen.setText(" Lade Eigenschaften aus " + properties + " ...", 20);
		props = initProperties( properties );
		
		setPLAF();	
		
		codeSnippets = loadCodeSnippets();
		wbPanel 	= new NekjeWBPanel( this );
		
		sscreen.setText(" Lade Plugins ...", 30);
		indentPlugin = wbPanel.loadIndentPlugin();
		referenceBrowser = initReferenceBrowser();
				
		sscreen.setText(" Konstruiere Menueleiste ...", 40);
		nekjeMenuBar 	= createMenuBar();
		
		sscreen.setText(" Konstruiere Toolleiste ...", 60);
		nekjeToolBar 	= createToolBar();
				
		addWindowListener(
			new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					wbPanel.closeAllTabs();
					System.exit(0);
				}
			}
		);
	
		sscreen.setText(" Fuege alle Teile zusammen ...", 80);
		setJMenuBar( nekjeMenuBar );
		getContentPane().setLayout( new BorderLayout() );		
		getContentPane().add( nekjeToolBar, BorderLayout.NORTH);
		getContentPane().add( wbPanel, BorderLayout.CENTER );
	
		sscreen.setText(" Setze Bildschirmpositionen ...", 90);
		
		setFrameSize();
		nekjeConsole = new NekjeConsole( this );
		


		sscreen.setVisible(false);
		sscreen.dispose();
		setVisible( true );
	}

	/**
	* Initialisert die integrierte Kontext-Hilfe
	*/
	protected NekjeReference initReferenceBrowser() {
		NekjeReference rBrowser = null;		
		String rData = getProperties().getProperty("referenceData");

		if( rData != null ) {
			rBrowser = new NekjeReference();
			rBrowser.setParent( this );
			rBrowser.setSize(400,400);
			rBrowser.setReferenceSource( rData );
			rBrowser.setClassPath( System.getProperty("java.class.path") );
		}
			
		return( rBrowser );	
	}	

	/**
	*	Setzt die Programm-Fenstergroesse 
	*/
	protected void setFrameSize() {
		int width = 600;
		int height = 400;
		int x = 0;
		int y = 0;

		String nWidth = getProperties().getProperty("nekjeWidth");
		String nHeight = getProperties().getProperty("nekjeHeight");	
		String nX = getProperties().getProperty("nekjeX");	
		String nY = getProperties().getProperty("nekjeY");	
		
		try {
			if( nWidth != null )
				width = Integer.parseInt( nWidth );
			if( nHeight != null )
				height = Integer.parseInt( nHeight );
			if( nX != null )
				x = Integer.parseInt( nX );
			if( nWidth != null )
				y = Integer.parseInt( nY );
		}	
		catch( NumberFormatException e ) {
			System.out.println("Keine gueltige Angabe: " + e.toString() );
		}
		setSize( width, height );
		setLocation( x, y );			
	
	}
	/**
	*	Setzt Look-and-Feel fuer die Applikation aufgrund der Eigenschaft "plaf".
	*	Die Eigenschaft kann in der Eigenschaftsdatei extern veraendert werden.
	*/
	public void setPLAF() {
	
		String plaf = getProperties().getProperty("plaf");
		
		if( plaf == null ) {
			plaf = UIManager.getSystemLookAndFeelClassName();
		}
		
		if( plaf != null ) {
			
			try {
				UIManager.setLookAndFeel( plaf );
			}
			catch( InstantiationException e ){
			}
			catch( ClassNotFoundException e ){
			}
			catch( UnsupportedLookAndFeelException e ){
			}
			catch( IllegalAccessException e ){
			}
		}			
	}
		
	public NekjeWBPanel getWBPanel() {
		return( wbPanel ); 
	}
	
	public NekjeConsole getConsole() {
		return( nekjeConsole );
	}
	
	
	public NekjeActionSet getActionSet() {
		return( wbPanel.getActionSet() );
	}
	
	private Properties initProperties( String pr ) {
	
		Properties tprops = new Properties();
		
		setDefaultProperties( tprops );
		loadProperties( tprops, pr );
		
		setSystemClassPath(tprops);
		
		return( tprops );	
	}
	
	private void setSystemClassPath( Properties tprops ) {
		String cp = tprops.getProperty( "classpath" );
		String wd = tprops.getProperty( "workingDirectory" );
		String od = tprops.getProperty( "outputDirectory" );
		
		String ps = tprops.getProperty( "pathSeparator" );
		String sp = System.getProperty( "java.class.path");
		
		if( cp != null && cp.length() > 0 ) {
			StringTokenizer st = new StringTokenizer( cp, ps);
			
			while( st.hasMoreTokens() ) {
				String temp = st.nextToken();
				if( sp.indexOf(temp) == -1 )
					sp = temp+ps+sp;
			}
		}
		
		/*
		if( wd != null && wd.length() > 0 && sp.indexOf( wd ) == -1 )
			sp += ps + wd;
		if( od != null && od.length() > 0 && sp.indexOf( od ) == -1 )
			sp += ps + od;
		*/
			
		if( sp.equals( System.getProperty("java.class.path") ) == false )
			System.setProperty("java.class.path", sp );						
	}
	
	private void setDefaultProperties( Properties pr ) {

		pr.setProperty("pathSeparator", System.getProperty("path.separator") );
		pr.setProperty("lineBreak", System.getProperty("line.separator") );		
		pr.setProperty("workingDirectory", System.getProperty("user.dir") );
		pr.setProperty("outputDirectory", "");
		pr.setProperty("classpath", "");
		pr.setProperty("nekjeFont", "Monospaced");
		pr.setProperty("nekjeFontsize", "14");
		pr.setProperty("nekjeFonttype", "BOLD");
		
	}
	
	public static Properties getProperties() {
		return( props );
	}
	
	public Vector getCodeSnippets() {
		return( codeSnippets );
	}
	
	private JToolBar createToolBar() {
		
			JToolBar toolB = new JToolBar();
			
			toolB.add( new SmallButton( getActionSet().actionNew ) );
			toolB.add( new SmallButton( getActionSet().actionOpen ) );
			toolB.add( new SmallButton( getActionSet().actionClose ) );
			toolB.add( getActionSet().actionResaveButton );
			toolB.addSeparator(); 
			toolB.add( new SmallButton( getActionSet().actionCut ) );
			toolB.add( new SmallButton( getActionSet().actionCopy) );
			toolB.add( new SmallButton( getActionSet().actionPaste) );
			toolB.addSeparator(); 
			toolB.add( new SmallButton( getActionSet().actionCloneCode) );
			toolB.add( new SmallButton( getActionSet().actionParagraph) );
			toolB.add( new SmallButton( getActionSet().actionComment) );


			if( getProperties().getProperty("appletviewerExe") != null ) {
				toolB.add( new SmallButton( getActionSet().actionAppletHTML) );
			}

			
			toolB.addSeparator(); 

		
			if( getProperties().getProperty("javacExe") != null ) {
				toolB.add( new SmallButton( getActionSet().actionCompile) );

			}

			
			if( getProperties().getProperty("javaExe") != null ) {
				toolB.add( new SmallButton( getActionSet().actionRunApp) );
			}
			
			if( getReferenceBrowser() != null ) {
				toolB.addSeparator(); 			
				toolB.add( new SmallButton( getActionSet().actionReference ) );
			}
			
			return( toolB );
		}	
		

	private JMenuBar createMenuBar() {
		
		Action[] dateiMenu = {
			getActionSet().actionNew,
			getActionSet().actionOpen,
			getActionSet().actionClose,
			getActionSet().actionResave,
			getActionSet().actionSave,
			getActionSet().actionPrint,
			null,
			getActionSet().actionExit
		};
		
		Action[] bearbeitenMenu = {
			getActionSet().actionCut,
			getActionSet().actionCopy,
			getActionSet().actionPaste,
			getActionSet().actionDelete,
			null,
			getActionSet().actionCloneCode,
			getActionSet().actionSelectAll
		};
			
		
		Object[][] mb = {
			{ "Datei" , dateiMenu },
			{ "Bearbeiten", bearbeitenMenu }
		};
		
		JMenuBar menuB = new JMenuBar();
		// Die Menue-Punkte der Menu-Leiste
		for( int i = 0; i < mb.length; i++ ) {
			
			JMenu tmpMenu = new JMenu( (String)mb[i][0] );
			
			// Dass zugehoerige SubMenu
			if( mb[i][1] != null ) {
				Action[] tptr = (Action[])mb[i][1];
				for( int j = 0; j < tptr.length; j++) {
					
					if( tptr[j] == null )
						tmpMenu.addSeparator();
					else {
						tmpMenu.add( (Action)tptr[j] );
					}
				}
				
			}
			menuB.add( tmpMenu );
		}
		
		// Konstruiere Tools-Menue
		if( getProperties() != null ) {
		
			JMenu tMenu = new JMenu("Tools");
			
			if( getProperties().getProperty("javacExe") != null ) {
				tMenu.add((Action)getActionSet().actionCompile);
			}
			if( getProperties().getProperty("javaExe") != null ) {
				tMenu.add((Action)getActionSet().actionRunApp);
				tMenu.add((Action)getActionSet().actionRunAppPara);
			}

			
			if( getProperties().getProperty("appletviewerExe") != null ) {
				tMenu.add((Action)getActionSet().actionAppletHTML);
			}
		
			if( indentPlugin == true ) {
				String indentPluginClass = 
					getProperties().getProperty("indentPluginClass");
				getActionSet().actionIndent.putValue( Action.NAME,
				"Code ordnen (" + indentPluginClass + ")" 
				);
				tMenu.add( getActionSet().actionIndent );
			}
			menuB.add( tMenu );
		}
		
		// Konstruiere Quelltext-Menu
		if( getCodeSnippets() != null ) {
			JMenu qMenu = new JMenu("Quelltext", true);
		
			for (Enumeration e = getCodeSnippets().elements() ; e.hasMoreElements() ;) {
				CodeSnippet snippet = (CodeSnippet)e.nextElement();
				String snippetKey = snippet.getKey();
				
				if( snippetKey.equals("Separator") ) {
					qMenu.addSeparator();
				}
				else {
					JMenuItem tItem = new JMenuItem( snippet.getKey() );
					tItem.setActionCommand( snippet.getKey() );
					tItem.addActionListener( getActionSet().actionCodeSnippet );
		
					qMenu.add(tItem);
				}
			}
 
			menuB.add( qMenu );

			menuB.add( Box.createHorizontalGlue() );

			
			if( getProperties().getProperty("docURL") != null && 

				getProperties().getProperty("browserExe") != null ) {

				JMenu menu_help = new JMenu("Hilfe");

				menu_help.add( getActionSet().actionViewDoc );
				menuB.add( menu_help );

			}

				
		}
		
		return( menuB );
	}		
	
	public void loadProperties( Properties pr, String resrc ) {
		try {
			URL src = new URL( resrc );
			pr.load( (InputStream)src.getContent() );
						
		}
		catch( Exception e ) {
			// Problem beim Laden der Resource
			try {
				pr.load( new FileInputStream(resrc) );
			}
			catch( Exception ex ) {			
				System.out.println( ex.toString() );
			}
			
		}	
	}
		
		
	public Vector loadCodeSnippets() {
		BufferedReader in = null;
		Vector vSnippets = null;

		// Besorge URL fuer Quelltext-Datei		
		String srcfile = getProperties().getProperty("codeSnippets");
		
		if( srcfile != null ) {
			
			try {
				URL src = new URL( srcfile );
				in = new BufferedReader( new InputStreamReader( (InputStream)src.getContent() ) );
			}
			catch( Exception e ) {
				try {
					in = new BufferedReader( new FileReader( srcfile) );
				}
				catch(Exception ex){
					System.out.println(ex.toString());
				}
			}
					
			if( in != null ) {
					
				String temp;
				String key = null;
				String value = "";

				vSnippets = new Vector();

				try {
					while( (temp = in.readLine()) != null ) {
						if( temp.startsWith("[") ) {
							if( key != null ) {
								vSnippets.addElement( new CodeSnippet( key, value ) );

								key = null;
								value = "";
							}
							
							if( key == null ) {
								key = temp.substring(1, 
									temp.indexOf(']'));	
							}								
						}
						else {
							value += temp + "\n";
						}						
					}
				}

				catch( IOException e ){
					// Fehler beim Einlesen der Datei
				}

					
				if( key != null ) {
					vSnippets.addElement( new CodeSnippet(key, value) );
				}

				
				try {
					in.close();
				}
				catch( IOException ex ){}
			}

		}
		
		return( vSnippets );
	}
	
	public String getCode( String key ) {
				for (Enumeration e = getCodeSnippets().elements() ; e.hasMoreElements() ;) {
					CodeSnippet snippet = (CodeSnippet)e.nextElement();
					
					if( key.equals(snippet.getKey()) )
						return( snippet.getValue() );
				}
				return(null);
	}
	
	public NekjeReference getReferenceBrowser() {
		return referenceBrowser;
	}
	
	
	/**
	* Hier kommt das Programm ins Rollen.
	*/
	public static void main( String[] args ) {

		String props = "nekje.cfg";
				
		// Suche nach Konfigurationsdatei
		if( args.length > 0 ) {
			props = args[0];	
		}
		
		new Nekje( props );		
	}
}
/**
* Klasse NekjeWBPanel
*
* Verwaltet den Workingbench der Anwendung
*/
class NekjeWBPanel extends JPanel {
	
	/**
	* Referenz auf das Tab-Panel des Workingbench-Panels
	*/
	protected JTabbedPane wbTabs;
		
	/**
	* Referenz auf alle verfuegbaren Editorfunktionen
	*/
	protected NekjeIndentClass indentObject;
	protected NekjeActionSet actionSet;
	private Nekje parentPanel;
	private int tabCounter;
	
	
	NekjeWBPanel( Nekje parent ) {
		parentPanel = parent;
		actionSet = new NekjeActionSet( parentPanel, this );
		setLayout( new BorderLayout() );
	
		wbTabs = new JTabbedPane();
		add(wbTabs, BorderLayout.CENTER);
		
		updateActionSet();
	}
	
	public NekjeScrollEditorPanel getEditor( int idx ) {
		return( (NekjeScrollEditorPanel)wbTabs.getComponentAt( idx ) );	
	}
	public NekjeScrollEditorPanel getCurrentEditor() {
		return( getEditor( wbTabs.getSelectedIndex() ) );
	}
	public NekjeActionSet getActionSet() {
		return( actionSet );
	}
	public void updateActionSet() {
		if( hasTab() ) {
			getActionSet().setEditActionsEnabled(true);
			getActionSet().setFileActionsEnabled(true);
			getActionSet().setToolActionsEnabled(true);
		}
		else {
			getActionSet().setEditActionsEnabled(false);
			getActionSet().setFileActionsEnabled(false);
			getActionSet().setToolActionsEnabled(false);
		}
		
	}
	public boolean hasTab() {
		return ( wbTabs.getTabCount() > 0 ? true : false );
	}
		
	public void addNewTab() {
		tabCounter++;
			
		String tempFileName = "Datei " + tabCounter;
		String workingDir = parentPanel.getProperties().getProperty("workingDirectory");
		
		NekjeScrollEditorPanel tp = 
			new NekjeScrollEditorPanel( this, workingDir, tempFileName, true );

		tp.setEditorFont(
			parentPanel.getProperties().getProperty("nekjeFont"),
			parentPanel.getProperties().getProperty("nekjeFonttype"),
			parentPanel.getProperties().getProperty("nekjeFontsize") );
						
		wbTabs.addTab( tempFileName, tp );
		wbTabs.setSelectedIndex( wbTabs.getTabCount()-1 );
		
		updateActionSet();
		tp.activateEditor();
	}
	

	public boolean changedDialog() {

		boolean status = true;
	
		if( getEditFlag() == true ) {
			status = saveBeforeActionDialog( 
				"Sie haben Vernderungen vorgenommen. Wollen Sie diese zunchst speichern?",
				getCurrentFileName() );
		}
		return(status);	

	}

	public void noJavaFile() {
			errorDialog( "Die gewhlte Funktion verlangt zwingend eine geffnete Datei\n" +
			"mit der Dateiendung .java im aktiven Karteifenster.");
	}
	
	public void errorDialog( String msg ) {
			JOptionPane.showMessageDialog( this, msg, "Fehler", JOptionPane.ERROR_MESSAGE); 
	}	

	public boolean isJavaFile() {
		boolean status = true;
		
		if(	getCurrentFileName().endsWith(".java") == false ) {
			JOptionPane.showMessageDialog( this, "Speichern Sie die Datei mit der Endung .java!",
			"Fehler", JOptionPane.ERROR_MESSAGE); 

			status = false;
		}
		
		return( status );
	}

	public boolean verifyFileName() {
		String fileName = getCurrentFileName();
		String className = fileName.substring( 0, fileName.lastIndexOf(".java") );
		 
		if( fileContainsClass( className ) == false ){
			errorDialog( "Die Datei " + fileName + " enthlt nicht die Klasse " + className);
			return( false );		
		} 
		
		return( true );
	}
		
	public boolean saveBeforeActionDialog(String msg, String title) {

		boolean status = false;

		Object[] possibleValues = { "Speichern", "Nicht Speichern", "Abbrechen" };

		int selectedValue = JOptionPane.showOptionDialog( 

					this, msg, title, JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE, 
					null, possibleValues, possibleValues[0] );

 		if( selectedValue == 0 )
 			status = resaveTab();

 		else if( selectedValue == 1 )
 			status = true;

		return( status );
	}

	public String getParameterDialog() {
		return( JOptionPane.showInputDialog(this, 
		"Welche Parameter wollen Sie an das Programm bergeben?") );	
	}
			 	

	public boolean closeSelectedTab( boolean save ) {
		
		boolean status = true;
	
		if( save )
			status = changedDialog();
			
		if( status == true ) {	
			wbTabs.removeTabAt( wbTabs.getSelectedIndex() );
			updateActionSet();
		}
		
		return(status);
	}

	public void closeTab( int idx, boolean save ) {
		
		wbTabs.setSelectedIndex( idx );
		closeSelectedTab( save );		
	}
	
	public boolean closeAllTabs() {
		boolean status = true;
		
		while( status == true && hasTab() ) {
			status = closeSelectedTab( true );
		}
		
		return( status );
	}
	
	public void cloneCode() {
		String orig = getCurrentEditor().getTextPane().getText();
		addNewTab();
		getCurrentEditor().getTextPane().setText( orig );
		getCurrentEditor().getTextPane().setCaretPosition( 0 );
	}
	
	public void openNewTab() {
  		    					
   		FileDialog fd = new FileDialog(parentPanel , "Datei ffnen",
							FileDialog.LOAD );
		fd.setDirectory( (String)parentPanel.getProperties().getProperty("workingDirectory") );
		fd.show();

 		if( fd.getFile() != null ) {    							
    			NekjeScrollEditorPanel temp = 
    					new NekjeScrollEditorPanel( NekjeWBPanel.this,
    					fd.getDirectory(), fd.getFile(), false );

		temp.setEditorFont(
			parentPanel.getProperties().getProperty("nekjeFont"),
			parentPanel.getProperties().getProperty("nekjeFonttype"),
			parentPanel.getProperties().getProperty("nekjeFontsize") );
			
			wbTabs.addTab( fd.getFile(), temp );	
			wbTabs.setSelectedIndex( wbTabs.getTabCount()-1 );
		
			updateActionSet();
			temp.activateEditor();

			// Setze Working-Directory neu
			parentPanel.getProperties().setProperty("workingDirectory", fd.getDirectory() );
    		}
    	}

	public boolean resaveTab( int idx ) {
		NekjeScrollEditorPanel nep = getEditor(idx);

		if( nep.getEditFlag() );
		{
			String cFile = getIndexedFileName( idx );
			String cName = nep.getClassName();

			if( nep.getSaveFlag() == false ) {
				if( cName.length() > 0 ) {
					return( saveDialog( idx, parentPanel.getProperties().getProperty("workingDirectory"),
							cName + ".java" ) );
				}
				else {
					return( saveDialog( idx, parentPanel.getProperties().getProperty("workingDirectory"),
						getIndexedFileName(idx) ) );
				}
			}
			else
				return( saveIndexedTab( idx, getIndexedDir(idx), getIndexedFileName(idx) ) );						
		}
	}
	
	public boolean resaveTab() {
			return( resaveTab( wbTabs.getSelectedIndex() ) );
	}
	
		
	/**
	* Zentrale Speicher-Funktion, durch die alle Speichervorgaenge laufen
	*/
	public boolean saveIndexedTab( int idx, String fDir, String fFile ) {
		
		boolean status = false;
		
		NekjeScrollEditorPanel temp = getEditor( idx );
		
		if( temp != null ) {
				
			try {
				temp.save( fDir, fFile );
				wbTabs.setTitleAt( idx, fFile );
				temp.setFFileName( fFile );
				temp.setFDirName( fDir );
				temp.setEditFlag( false );
			}
			catch( IOException e ) {
				return(false);
			}
			
			status = true;		
		}			
		else
			status = false;
			
		return( status );
	}
	
	public void saveCurrentTab( String fDir, String fFile ) {
		saveIndexedTab( wbTabs.getSelectedIndex(), fDir, fFile );
	}
		
	public boolean getEditFlag() {
		if( wbTabs.getTabCount() > 0 ) {		
			NekjeScrollEditorPanel temp = getCurrentEditor();
		
			return( temp.getEditFlag() );
		}
		return(false);
	}
	public void deleteSelection() {
			getCurrentEditor().getTextPane().replaceSelection("");
		}
	public void replaceSelection( String text ) {
			getCurrentEditor().getTextPane().replaceSelection(text);		
	}
			
	public void selectAll() {
			getCurrentEditor().getTextPane().selectAll();
	}
	
	public void insertParagraph() {
		
		getCurrentEditor().insertParagraph();
	}
	
	public void insertBlock( String text, int newpos ) {
		getCurrentEditor().insertBlock( text, newpos );
	}
	
	public String getCurrentTextToken() {
		return(getCurrentEditor().getCaretText());
	}
	
	public boolean loadIndentPlugin() {
		
		boolean status = false;
		

		String indentPluginURL = 
		parentPanel.getProperties().getProperty("indentPluginURL");
		String indentPluginClass = 
		parentPanel.getProperties().getProperty("indentPluginClass");
		
		
		if( indentPluginURL != null && indentPluginClass != null ) {
			
		
			try {
				// Wrapper-Klassen muessen den Klassennamen + "PluginWrapper" tragen!!!!
				String indentPluginName = "";
				
				try {
					// Trenne classpath von Class-Name
					indentPluginName = indentPluginClass.substring( 
					indentPluginClass.lastIndexOf('.')+1 );
				}
				catch( IndexOutOfBoundsException eb ) {
					indentPluginName = indentPluginClass;
				}
				
				if( indentPluginName.length() > 0 ) {
					String indentPluginWrapper = indentPluginName+"PluginWrapper"; 
					
					Class indentClass = Class.forName(indentPluginWrapper);
				
					this.indentObject = 
						(NekjeIndentClass)indentClass.newInstance();
					this.indentObject.setURL( indentPluginURL );
					this.indentObject.setClassName( indentPluginClass );
					this.indentObject.loadPlugin();
					
					status = this.indentObject.isValid();
				}
			}
			catch( Exception e ) {
				System.err.println( e.toString() );
				status = false;
			}		
		}
		
		if( status == false && this.indentObject != null )
			indentObject = null;

		return(status);
	}
	
	public void indent() {
		
		if( indentObject != null ) {	
			Document doc = getCurrentDocument();
		
			try {
				String text = doc.getText(0, doc.getLength());
				    				
 				ByteArrayInputStream in = new ByteArrayInputStream(text.getBytes());
      				ByteArrayOutputStream out = new ByteArrayOutputStream();
      		
      				indentObject.parse(in,out);

     				text = out.toString();

				selectAll();
				replaceSelection(text);
			}
			catch( Exception e ) {
			}     			
 		}
 	}

	
	public void createHTML( String dirName, String className ) {

		String text = "<HTML>\n<!-- HTML-Testdatei fuer Java-Applet " + className + "-->\n" +
			"<BODY>\n<APPLET CODE=\"" + className + ".class\" " +
			"WIDTH=\"400\" HEIGHT=\"200\">\n</APPLET>\n</BODY></HTML>\n";

		String tFName = className+ ".htm";
		int idx = wbTabs.getSelectedIndex();
		
		NekjeScrollEditorPanel temp = getEditor( idx );

		wbTabs.setTitleAt( idx, tFName );
		temp.setFFileName( tFName );
		temp.setFDirName( dirName );
		getCurrentEditor().insertBlock( text, 0 );
	}

	public String getCurrentFile() {
		NekjeScrollEditorPanel temp = getCurrentEditor();
	
		return( temp.getFDirName() + temp.getFFileName() );
	}
	
	public String getCurrentDir() {
		NekjeScrollEditorPanel temp = getCurrentEditor();
	
		return( temp.getFDirName() );
	}
	public String getCurrentFileName() {
		NekjeScrollEditorPanel temp = getCurrentEditor();
	
		return( temp.getFFileName() );
	}
	public String getIndexedFile( int idx ) {
		NekjeScrollEditorPanel temp = getEditor(idx);
	
		return( temp.getFDirName() + temp.getFFileName() );
	}
	public String getIndexedDir( int idx ) {
		NekjeScrollEditorPanel temp = getEditor( idx );
	
		return( temp.getFDirName() );
	}
	public String getIndexedFileName( int idx ) {
		NekjeScrollEditorPanel temp = getEditor( idx );
	
		return( temp.getFFileName() );
	}
	public Document getCurrentDocument() {
		return( getCurrentEditor().getDocument() );
	}
	
	public void changedUpdate() {
		wbTabs.setTitleAt(wbTabs.getSelectedIndex(), 
		wbTabs.getTitleAt(wbTabs.getSelectedIndex()) + "*" );	
	}
	

	public void saveAs() {
	
		saveDialog( wbTabs.getSelectedIndex(), 
			parentPanel.getProperties().getProperty("workingDirectory"), 
			getCurrentFileName()  );
	}
	

	public String getClassName() {

		return(getCurrentEditor().getClassName());

	}

	public boolean fileContainsClass( String className ) {
		return( getCurrentEditor().fileContainsClass( className ) );
	}
	
	
	private int addJavaExtensionDialog( String fname ) {
		Object[] possibleValues = { "Ja", "Nein", "Abbrechen" };

		return JOptionPane.showOptionDialog( 
			this, "Soll die Dateiendung \".java\" ergnzt werden",
			"Warnung: " + fname + " - fehlende Dateiendung", 
			JOptionPane.YES_NO_CANCEL_OPTION, 
			JOptionPane.WARNING_MESSAGE, 
			null, possibleValues, possibleValues[0] );

 	}

	public boolean saveDialog( final int idx, final String fdir, final String fname ) {

   		boolean status = false;
   		boolean dialogDone = false;
  		String fileName = fname;
  		String fileDir = fdir;	
  		   					
	    	while( dialogDone != true ) {
	    		final FileDialog fd = new FileDialog( parentPanel, "Datei speichern", 
    			   			FileDialog.SAVE );
    			fd.setDirectory( fileDir );
    			fd.setFile( fileName );
    			fd.show();
 			
 			fileName = fd.getFile();
 			fileDir = fd.getDirectory();			
 			dialogDone = true;
 			
 			if( fileName != null ) {
 				// Check die Endung
 				if( fileName.indexOf('.') == -1 ) {
 					// Keine Dateiendung entdeckt
 					int rv = addJavaExtensionDialog( fileName );
 					
 					if( rv == 0 ){
 						fileName = fileName + ".java";
 					}
 					else if( rv == 2 ) {
 						dialogDone = false;	
 					}
 				}					
 			}	
 		}
 			
 		if( fileName != null ) {				    							
	    		status = saveIndexedTab( idx, fileDir, fileName );

    			if(status == true)
			{
    				parentPanel.getProperties().setProperty("workingDirectory", 
    				fileDir );
    			}
    		}

		return(status);
	}
	
	public void processCommand( final String command ) {
		Thread runner = new Thread() {
			public void run() {
				try {
					Process p = Runtime.getRuntime().exec(command);
				}
				catch( IOException ex ) {
				}
			}
		};
		runner.start();
	}
	

	public void printEditorText() {
		
		final Document doc = getCurrentDocument();
		
		// Get a PrinterJob
    		final PrinterJob job = PrinterJob.getPrinterJob();
    		// Create a landscape page format
    		PageFormat pf = job.defaultPage();
    		
 		Book bk = new Book();
   		DocumentPainter dp = new DocumentPainter( doc, pf, true );
   		int pnum = dp.calculatePageCount();
       		bk.append( dp, pf, pnum );
   		
    		// Pass the book to the PrinterJob
    		job.setPageable( bk );
    		
    		
		// Put up the dialog box
		if (job.printDialog()) {
       			// Print the job if the user didn't cancel printing
      			Thread runner = new Thread() {
      			
      				public void run() {
      					try { job.print(); }
      					catch (Exception exc) {}	      				
      					}
    			};
    			
    			runner.start();
		}	
    }

}

/**
* Klasse NekjeActionSet definiert alle Handlungen, die vom Menue und von der
* Tool-Leiste ausgefuehrt werden koennen.
*/
class NekjeActionSet {
	
	public Action actionNew;
	public Action actionOpen;
	public Action actionResave;
	public Action actionSave;
	public Action actionPrint;
	public Action actionClose;
	public Action actionExit;
		
	public Action actionCut;
	public Action actionCopy;
	public Action actionPaste;
	public Action actionDelete;
	public Action actionSelectAll;
	
	public Action actionCloneCode;
	
	public Action actionParagraph;
	public Action actionComment;
	public Action actionCodeSnippet;
	
	public Action actionCompile;
	public Action actionRunApp;
	public Action actionRunAppPara;
	public Action actionIndent;

	public Action actionAppletHTML;

	public Action actionViewDoc;
	public Action actionReference;
	
	
	public SmallButton actionResaveButton;
		
	protected final Nekje ppanel;
	protected final NekjeWBPanel wbpanel;
	
	public NekjeActionSet( Nekje pp, NekjeWBPanel wb ) {
		
		ppanel = pp;
		wbpanel = wb;
		
		actionNew = new AbstractAction("Neu", new ImageIcon( 
			getClass().getResource("images/file_new.gif") ) ) {	
			public void actionPerformed(ActionEvent e) {
      				wbpanel.addNewTab();
			}
		};
		actionNew.putValue(Action.SHORT_DESCRIPTION, new String("Neues Editorfenster")); 
   			
		actionOpen = new AbstractAction( "ffnen...", new ImageIcon( 
			getClass().getResource("images/file_open.gif") ) ) {
   			public void actionPerformed( ActionEvent e ) {
   					wbpanel.openNewTab();
    		}
    	};
    	actionOpen.putValue(Action.SHORT_DESCRIPTION, new String("Datei ffnen...")); 
    			
    	actionResave = new AbstractAction("Speichern", new ImageIcon(
    		getClass().getResource("images/file_save.gif"))) {
    		public void actionPerformed( ActionEvent e ) {	
    			
   			if( wbpanel.hasTab() )
				wbpanel.resaveTab();
			}   			    			
    	};
    	actionResave.putValue(Action.SHORT_DESCRIPTION, new String("Speichern")); 
    			
    	actionResaveButton = new SmallButton( actionResave );
    			
    	actionSave = new AbstractAction( "Speichern unter...", null) {
    		public void actionPerformed( ActionEvent e ) {
    			if( wbpanel.hasTab() )
    				wbpanel.saveAs();
    		}
    	};
    		
      	actionPrint = new AbstractAction( "Drucken...", null) {
      		public void actionPerformed( ActionEvent e ) {
    			if( wbpanel.hasTab() )
    				wbpanel.printEditorText();
    			else
    				Toolkit.getDefaultToolkit().beep();
     		}
    	};
   			
	actionClose = new AbstractAction("Schliessen", 
		new ImageIcon( getClass().getResource("images/file_close.gif") ) ) {	
			public void actionPerformed(ActionEvent e) {
				if( wbpanel.hasTab() )
       					wbpanel.closeSelectedTab( true );
      			}
	};		
	actionClose.putValue(Action.SHORT_DESCRIPTION, new String("Fenster schlieen")); 
   			
	actionExit = new AbstractAction("Beenden", null) {
		public void actionPerformed(ActionEvent e) {
			if( wbpanel.closeAllTabs() == true )				
				System.exit(0);
		}
   	};
 			
   	actionCut = new DefaultEditorKit.CutAction();
   	actionCut.putValue(Action.SMALL_ICON, new ImageIcon(
   		getClass().getResource("images/edit_cut.gif")) );
   	actionCut.putValue(Action.SHORT_DESCRIPTION, new String("Ausschneiden")); 
   	actionCut.putValue(Action.NAME, new String("Ausschneiden")); 
    		
   	actionCopy = new DefaultEditorKit.CopyAction();
   	actionCopy.putValue(Action.SMALL_ICON, new ImageIcon(
   		getClass().getResource("images/edit_copy.gif")) );
   	actionCopy.putValue(Action.SHORT_DESCRIPTION, new String("Kopieren")); 
   	actionCopy.putValue(Action.NAME, new String("Kopieren")); 
    		
    			
	actionPaste = new DefaultEditorKit.PasteAction();
	actionPaste.putValue(Action.SMALL_ICON, new ImageIcon(
		getClass().getResource("images/edit_paste.gif")) );
	actionPaste.putValue(Action.SHORT_DESCRIPTION, new String("Einfgen")); 
	actionPaste.putValue(Action.NAME, new String("Einfgen")); 
	actionDelete = new AbstractAction("Lschen", null) {
		public void actionPerformed(ActionEvent e) {
   			wbpanel.deleteSelection();
   		}
   	};
 
   	actionSelectAll = new AbstractAction("Alles markieren", null) {
   		public void actionPerformed(ActionEvent e) {
   				wbpanel.selectAll();
   		}
   	};
		
	actionCloneCode = new AbstractAction("Quelltext duplizieren", 
		new ImageIcon(getClass().getResource("images/edit_dup.gif")) ) {
		public void actionPerformed( ActionEvent e ) {
			if( wbpanel.hasTab() == true ) {
				wbpanel.cloneCode();
			}
		}
	};

   	actionCloneCode.putValue(Action.SHORT_DESCRIPTION, new String("Quelltext duplizieren")); 
	
	actionParagraph = 
		new AbstractAction("Block einfgen", new ImageIcon(
			getClass().getResource("images/edit_para.gif")) ) {
			public void actionPerformed( ActionEvent e ) {
				if( wbpanel.hasTab() == true ) {
					wbpanel.insertParagraph();
			}
		}
	};
	actionParagraph.putValue(Action.SHORT_DESCRIPTION, new String("Codeblock einfgen")); 
	
	actionComment = new AbstractAction("Kommentar einfgen", new ImageIcon(
		getClass().getResource("images/edit_comment.gif")) ) {
		public void actionPerformed( ActionEvent e ) {
			if( wbpanel.hasTab() == true ) {
				wbpanel.insertBlock("/**\n*\t\n*/\n", 6);
			}
		}
	};
	actionComment.putValue(Action.SHORT_DESCRIPTION, new String("Kommentar einfgen")); 
			
	actionCodeSnippet = new AbstractAction("", null ) {
		public void actionPerformed( ActionEvent e ) {
			
			String command = e.getActionCommand();
			String code = (String)ppanel.getCode(command);
			
			if( code != null ) {
				if( wbpanel.hasTab() == false ) {
					wbpanel.addNewTab();
				}
					
				if( wbpanel.hasTab() ) {	
					wbpanel.insertBlock( code, -1 );			
				}
			}
		}
	};
			
	actionCompile = new AbstractAction("Kompilieren", new ImageIcon(
			getClass().getResource("images/tool_com.gif")) ) {
		public void actionPerformed( ActionEvent e ) {
			if( wbpanel.hasTab() ) {
				if( wbpanel.isJavaFile() && wbpanel.changedDialog() && wbpanel.verifyFileName() ) {
					ppanel.getConsole().compileJavaApp( 
					wbpanel.getCurrentDir(),
					wbpanel.getCurrentDir(),
					wbpanel.getCurrentFileName() );
				}
			}
			else
				wbpanel.noJavaFile();
		}
	};
		
	actionCompile.putValue(Action.SHORT_DESCRIPTION, new String("Kompilieren")); 

	actionRunApp = new AbstractAction("Ausfhren", new ImageIcon(
		getClass().getResource("images/tool_run.gif")) ) {
		public void actionPerformed( ActionEvent e ) {
			if( wbpanel.hasTab() == true ) {						
				ppanel.getConsole().runJavaApp(
					wbpanel.getCurrentDir(),
					wbpanel.getCurrentFileName(),
					null, false );
			}
			else
				wbpanel.noJavaFile();
		}	
	};
	actionRunApp.putValue(Action.SHORT_DESCRIPTION, new String("Ausfhren")); 

	actionRunAppPara = new AbstractAction("Ausfhren mit Parametern", null ) {
		public void actionPerformed( ActionEvent e ) {
			if( wbpanel.hasTab() == true ) {
				String clParameter = wbpanel.getParameterDialog();
				
				if( clParameter != null ) {
					ppanel.getConsole().runJavaApp(
					wbpanel.getCurrentDir(),
					wbpanel.getCurrentFileName(),
					clParameter, false );
				}
			}
			else
				wbpanel.noJavaFile();
		}	
	};


	actionAppletHTML = new AbstractAction("HTML-Datei generieren", 
		new ImageIcon(getClass().getResource("images/edit_html.gif")) ) {
		public void actionPerformed( ActionEvent e ) {

			String appletName = "xy";
			String dirName = ppanel.getProperties().getProperty("workingDirectory");
			
			if( wbpanel.hasTab() ) {
				if( wbpanel.isJavaFile() )
				{
					dirName = wbpanel.getCurrentDir();
					appletName = wbpanel.getClassName();
							
					wbpanel.addNewTab();
					wbpanel.createHTML( dirName, appletName );
				}
			}
			else
				wbpanel.noJavaFile();
		}	
	};
	actionAppletHTML.putValue(Action.SHORT_DESCRIPTION, new String("HTML erzeugen")); 

	actionIndent = new AbstractAction("Quellcode ordnen", null) {
	
		public void actionPerformed( ActionEvent e ) {
			if( wbpanel.hasTab() ) {
				wbpanel.indent();
			}
			else
				wbpanel.noJavaFile();
			
		}
			
	};	

	actionViewDoc = new AbstractAction("Online-Dokumentation", null ) {
		public void actionPerformed( ActionEvent e ) {

			wbpanel.processCommand( ppanel.getProperties().getProperty("browserExe") +
				" " + ppanel.getProperties().getProperty("docURL") );
		}	
	};
		
	actionReference = new AbstractAction("Klassen-Referenz", 
		new ImageIcon( getClass().getResource("images/help.gif") ) ) {
		public void actionPerformed( ActionEvent e ) {
			
			Thread runner = new Thread() {
				
				public void run() {
			
					if( wbpanel.hasTab() ) {
						String currentText = wbpanel.getCurrentTextToken();

						if( currentText.length() > 0 ) {
							ppanel.getReferenceBrowser().getReference( currentText );
						}
					}
					else
						ppanel.getReferenceBrowser().setAusgabeText("");
						
					ppanel.getReferenceBrowser().setVisible(true);
				}
			};
			
			runner.start();
					
		}	
	};
	actionReference.putValue(Action.SHORT_DESCRIPTION, new String("Referenz anzeigen")); 

	}

	public void setEditActionsEnabled( boolean value ) {
		actionCut.setEnabled( value );
		actionCopy.setEnabled( value );
		actionPaste.setEnabled( value );
		actionSelectAll.setEnabled( value );
		actionDelete.setEnabled( value );
		actionCloneCode.setEnabled( value );
	}
	
	public void setToolActionsEnabled( boolean value ) {
		actionCompile.setEnabled( value );
		actionRunApp.setEnabled( value );
		actionRunAppPara.setEnabled( value );
		actionAppletHTML.setEnabled( value );
		actionIndent.setEnabled( value );
	}
	public void setFileActionsEnabled( boolean value ) {
		actionSave.setEnabled( value );
		actionResave.setEnabled( value );
		actionClose.setEnabled( value );
		actionPrint.setEnabled( value);
	}
}

class NekjeScrollEditorPanel extends JPanel implements DocumentListener {
	
	protected NekjeWBPanel parentPane;
	protected NekjeEditorPane tp;
	protected JPanel statusPanel;
	protected JLabel rcLabel;
	protected JLabel statusLabel;
	protected JScrollPane scrollPane;
	protected Document doc;
	protected String fFileName;
	protected String fDir;
	protected boolean editFlag; 
	protected boolean saveFlag;

	
	public NekjeScrollEditorPanel( NekjeWBPanel parent, String fDirec, String fFile,
		boolean blankWindow ) {
		
		// Uebernahme der Parameter
		parentPane = parent;
		fFileName = fFile;
		fDir = fDirec;
		
		tp = new NekjeEditorPane();
		
		if( !blankWindow && tp != null && fFile != null && fFile.length() > 0 ) {
			
			try {
				String filePath = (fDir != null ? fDir : ".") + fFile;
					
				FileReader in = new FileReader( new File( filePath ) );	
				
				tp.read( in, null );
				in.close();	

				setSaveFlag(true);	
			}
			catch( IOException e ) {
				
			}
		}
		else

			setSaveFlag(false);	
	

		// Editorfenster initialisieren						
		scrollPane = new JScrollPane( tp );
		
		// Statuszeile konstruieren
		statusPanel = new JPanel();
		
		rcLabel = new JLabel(" Zeile: 1 Position: 1");
		statusLabel = new JLabel();
		
		
		statusPanel.setLayout( new BorderLayout() );
		statusPanel.add(rcLabel, BorderLayout.WEST);
		statusPanel.add(statusLabel, BorderLayout.EAST);
		
		// EditorPanel konstruieren
		setLayout( new BorderLayout() );
		add(scrollPane, BorderLayout.CENTER );
		add(statusPanel, BorderLayout.SOUTH);
		
		doc = tp.getDocument();
		doc.putProperty( PlainDocument.tabSizeAttribute, new Integer(4) );
		doc.addDocumentListener( this );
		tp.addCaretListener( new CaretListener() {
			public void caretUpdate( CaretEvent e ) {
	
					Element elem = doc.getDefaultRootElement();
					int pos = tp.getCaretPosition();
					int idx = elem.getElementIndex(pos);
					Element cElem = elem.getElement(idx);
					int start = cElem.getStartOffset();
									
					rcLabel.setText(" Zeile: " + (idx+1) + " Position: " + (pos-start+1) );
			
			}
		} );
	}
	
	/**
	*	Setzt den Font fuer das Editorfenster.
	*/
	public void setEditorFont( Font f ) {
		tp.setFont( f );
	}
	
	public void setEditorFont( String font, String fontType, String fontSize ) {
		
		int fsize=14;
		int ftype = Font.PLAIN;
		
		try {
			fsize = Integer.parseInt( fontSize );
			}
		catch(NumberFormatException e) {}
		
		if( fontType.equals("BOLD") ) { ftype = Font.BOLD; }
		else if( fontType.equals("ITALIC") ) { ftype = Font.ITALIC; }		
		
		tp.setFont( new Font( font, ftype, fsize ) );
	}
	
	
	/**
	* LowLevel-Datei-Speichern
	*/
	public void save( String fDir, String fFile ) throws IOException {
			
			FileWriter out = new FileWriter( new File(fDir + fFile) );
			tp.write( out );
			out.close();

			setSaveFlag(true);

	}
	
	public void activateEditor() {
		tp.requestFocus();
	}
	
	
	/**
	*	Fuegt den String text an der aktuellen Stelle in das Dokument ein
	*/
	public void insertString( int pos, String text ) {
	
		try {
			doc.insertString(pos, text, null);
		}
		catch( BadLocationException e ) {}
	}
	
	public NekjeEditorPane getTextPane() {
		return( tp );
	}
	
	public String getCaretText() {
		int pos = tp.getCaret().getDot();
		NekjeDocument tdoc = (NekjeDocument)getDocument();
		
		return( tdoc.getCurrentToken( pos ) );
	}
	
	public void insertParagraph() {
		
		int pos = tp.getCaret().getDot();
		NekjeDocument tdoc = (NekjeDocument)getDocument();
		
		int newpos = tdoc.insertParagraph( pos );
		
		tp.setCaretPosition( newpos );
	}
	
	public void insertBlock( String text, int newpos ) {
		
		int pos = tp.getCaret().getDot();
		NekjeDocument tdoc = (NekjeDocument)getDocument();
		
		int npos = tdoc.insertBlock( text, pos, newpos );
		
		tp.setCaretPosition( npos );
	}
	
	public void setFFileName( String name ) {
		fFileName = name; 
	}
	public String getFFileName() {
		return( fFileName );
	}
	public void setFDirName( String name ) {
		fDir = name;
	}
	public String getFDirName() {
		return( fDir );
	}
	public void setEditFlag( boolean flag ) {
		editFlag = flag;
	}
	
	public boolean getEditFlag() {
		return( editFlag );
	}
	
	public void setSaveFlag( boolean flag ) {
		saveFlag = flag;
	}

	public boolean getSaveFlag() {
		return( saveFlag );
	}
	
	public Document getDocument() {
		return( doc );
	}
	
	public String getClassName() {

		NekjeDocument cd = (NekjeDocument)getDocument();
		return( cd.getClassName() );	

	}


	public boolean fileContainsClass( String classname ) {
		NekjeDocument cd = (NekjeDocument)getDocument();
		return( cd.documentContainsClass( classname ) );
	}
			
	public void changedUpdate(DocumentEvent e) {
		
		if( getEditFlag() == false ) {
			setEditFlag(true);
			parentPane.changedUpdate();
		}
	}
	
	public void insertUpdate(DocumentEvent e) {
		changedUpdate( e );
	}
	
	public void removeUpdate(DocumentEvent e) {
		changedUpdate( e );
	}

}
   
   
class NekjeEditorPane extends JEditorPane {

	/**
	 * The default editor kit for this text component.
	 */
	public static final EditorKit EDITOR_KIT = new NekjeEditorKit();

	

	/**
	 * Creates a new SyntaxTextArea component.
	 */
	public NekjeEditorPane()
	{
		super();		
	}
	/**
	 * Returns the default editor kit for this text component.
	 */
	public EditorKit createDefaultEditorKit()
	{
		return EDITOR_KIT;
	}
	

                
}

class NekjeEditorKit extends DefaultEditorKit implements ViewFactory {
	
	public ViewFactory getViewFactory()
	{
		return this;
	}
	/**
	 * Creates a view from an element that can be used for painting that
	 * element. This implementation returns a new <code>PlainView</code>
	 * instance.
	 * @param elem The element
	 * @see org.gjt.sp.jedit.syntax.SyntaxView
	 */
	public View create(Element elem)
	{
		return new NekjeView(elem);
	}
	/**
	 * Creates a new instance of the default document for this
	 * editor kit. This returns a new instance of
	 * <code>NekjeDocument</code>.
	 * @see org.gjt.sp.jedit.syntax.DefaultSyntaxDocument
	 */
	public Document createDefaultDocument() {
		return( new NekjeDocument() );
	}
}

class NekjeView extends PlainView
{
	Segment line;

	
	public NekjeView( Element elem ) {
		super( elem );
		line = new Segment();

	}

	public void drawLine(int lineIndex, Graphics g, int x, int y)
	{
		NekjeDocument document = (NekjeDocument)getDocument();
		FontMetrics metrics = g.getFontMetrics();
		Color def = Color.black;
		int offset = 0;
		g.setColor( def );
		try
		{
			Element lineElement = getElement().getElement(lineIndex);
			int start = lineElement.getStartOffset();
			int end = lineElement.getEndOffset();
			
			// Lese die gesamte Zeile in ein Segment-Array
			document.getText(start, end-start, line);
			x = Utilities.drawTabbedText(line,x,y,g,this,offset);			
		}
		catch(BadLocationException bl)
		{
			// shouldn't happen
			
		}
	}
}

class NekjeDocument extends PlainDocument {
	
	public NekjeDocument() {
		super();
	}
	
	public int insertParagraph( int offset ) {
	
		String indent = getRowIndent( getRowStart(offset), offset );

		String lb = "\n";
		int ilen = indent.length();
		
		try {
			insertString( offset, "{" + lb + indent + "\t" + lb + indent + "}", null );
			return( offset + ilen + 3 );
		}
		catch( BadLocationException e ) {

		}

		return( offset );
	}


	public int insertBlock( String text, int offset, int newpos ) {
	
		int position = 0;
		int maxpos = newpos;
		
//		String lb = "\n";
		StringTokenizer stok = new StringTokenizer( text, "\n", true);
		
		// Holt Einzug der aktuellen Zeile
		String indent = getRowIndent( getRowStart(offset), offset );
		int ilen = indent.length();
                		
   		String newtext = "";
			
		while( stok.hasMoreTokens() ) {
			String temp = stok.nextToken()/* + lb */;
			int len = temp.length();
			
			newtext += temp;
			
			if( maxpos > 0 ) {
				if( maxpos - len >= 0 ) {
					position += len;
					maxpos -= len;
				}
				else {
					position += maxpos;
					maxpos = -1;
				}
			}
			
			if( temp.endsWith("\n") ) {
				newtext += indent;
				if( maxpos >= 0 )
					position += indent.length();
			}
			
			if( maxpos == 0) 
				maxpos = -1;
		}
		
		try {
			insertString( offset, newtext, null );
		}
		catch( BadLocationException e ) {}
		
		return( offset+position );
	}
	
	public void insertString(int offset, String str, AttributeSet a)
           throws BadLocationException {
                
    	int start;
        String lb = "\n";
                	
        if( str.equals( lb ) ) {
                		
        	start = getRowStart( offset );
                		
        	if( start < offset ) {
           		if( lb != null )
    	   			str = lb;
                				
       			str += getRowIndent( start, offset  );
       		}
       	}
       	else if( str.equals("{") ) {
			start = getRowStart( offset );
              		
    		if( start < offset ) {
   		   		str += lb +"\t" + getRowIndent( start, offset  );
    		}
    		
      	}
       	else if( str.equals("}") ) {
       		int tstart = getRowStart(offset);
       		String temp = getText( tstart, offset-tstart ) ;
       		String prev = getPreviousIndent( offset);
                    		
      		if( isWhitespaceString( temp ) ){
				if( temp != null && temp.length() > 0 ) {
					remove( tstart, offset-tstart);
					offset = tstart;
				}
				str = prev + "}" + lb + prev;	
			}
       		else {        		
       			str = lb + prev + "}" + lb + prev;
       		}
       	}
                		
		super.insertString( offset, str, a );                
	}
                
                
        public String getCurrentToken( int offset ) {
        	
        	int rowStart = getRowStart( offset );
        	int rowEnd = getRowEnd( offset );
        	String memToken="";
        	String currentToken = "";
        	int currentPos = 0;
        	int searchPos = offset-rowStart;
        	
        	try {
          		String temp = getText( rowStart, rowEnd-rowStart );
      	
        		StringTokenizer st = 
        		new StringTokenizer(temp, " ;,.(){}[]=<>!%\"\t\n\r\f");
        	
			while (st.hasMoreTokens()) {

				currentToken = st.nextToken();
     		
     				currentPos = temp.indexOf( currentToken, currentPos );

     				if( currentPos > searchPos )
     					break;
     					
     				memToken = currentToken;	
     			}
     		}
     		catch( BadLocationException e ) {
     			memToken = "";
     		}
     		
     		return( memToken );
        }
        
	/**
	*	Gibt die Startposition der aktuellen Zeile im Dokument zurueck
	*/
	public int getRowStart( int offset ) {		
		return( getParagraphElement( offset).getStartOffset() );
	}
	
	public int getRowEnd( int offset ) {		
		return( getParagraphElement( offset).getEndOffset() );
	}

	
	public boolean isWhitespaceString( String str ) {
	
		for( int i = 0; i < str.length(); i++ ) {
			if( Character.isWhitespace( str.charAt(i) ) == false )
				return( false );
		}
		
		return( true );		
	}
	
	public String getClassName() {

		String className = "";
		ElementIterator eIt = new ElementIterator( this );
		Element cElement = null;

		boolean found = false;
		
		for( cElement = eIt.first(); 
			cElement != null && (found == false || className.length() < 1); 
			cElement = eIt.next() ) {

			int start = cElement.getStartOffset();
			int end = cElement.getEndOffset();
			
			try{
				StringTokenizer sTokens = new StringTokenizer( getText( start, end-start ),
					" {}\t\n\r\f" );

				while( sTokens.hasMoreTokens() ) {
					String str = sTokens.nextToken();

					if( found == true )	{
						className = str;
						break;
					}

					if( str.equals("class") || str.equals("interface") )
						found = true;				
				}
			}

			catch( BadLocationException e ) {
				 break; 
			}
		}

		return( className );
	}

	public boolean documentContainsClass( String classname ) {

		ElementIterator eIt = new ElementIterator( this );
		Element cElement = null;

		boolean found = false;
		boolean classFound = false;
		
		for( cElement = eIt.first(); 
			cElement != null && (found == false || classFound == false); 
			cElement = eIt.next() ) {

			int start = cElement.getStartOffset();
			int end = cElement.getEndOffset();
			
			try{
				StringTokenizer sTokens = new StringTokenizer( getText( start, end-start ),
					" {}\t\n\r\f" );

				while( sTokens.hasMoreTokens() ) {
					String str = sTokens.nextToken();

					if( found == true )	{
						if( str.equals(classname) )
						{
							classFound = true;
							break;
						}
						else{
							found = false;
							classFound = false;
						}
					}
					
					if( str.equals( "class" ) || str.equals( "interface" ) )
						found = true;				
				}
			}

			catch( BadLocationException e ) {
				 break; 
			}
		}

		return( found && classFound );
	}


	public String getPreviousIndent(int offset) {
	
		Element cElement = getDefaultRootElement();
		int cIndex = cElement.getElementIndex( offset );
		
		int ebene = 0;
		
		for(int i = cIndex; i >= 0; i--) {
			Element tElement = cElement.getElement( i );
			int start = tElement.getStartOffset();
			int end = tElement.getEndOffset();
			
			try{
				String elemString = getText( start, end-start );
				if( elemString.indexOf( (int)'{') >= 0 ) {
					
					if( ebene <= 0 )
						return( getRowIndent(tElement.getStartOffset(), 
						tElement.getEndOffset()) );
					else
						ebene--;	
				}
				else if( elemString.indexOf( (int)'}') >= 0 ) {
					ebene++;
				}	
			}
			catch( BadLocationException e ) { 
				break; 

			}
		}
		return("");
	}
	
	/**
	*	Liefert als String den Beginn der aktuellen Zeile, wenn Einzug vorliegt
	*/
	public String getRowIndent( int start, int offset ) {
		
		String elemString = null;
		int charPos = 0;
		
		try {
			elemString = getText(start, offset-start);
		
			for( ; charPos < (offset-start) &&
					(elemString.charAt(charPos) == '\t' ||
					elemString.charAt(charPos) == ' ' ); charPos++ ) ;
		}
		catch( BadLocationException e ) {
				charPos = 0;
		}
		return( elemString.substring( 0, charPos ) );
	}
}

/**
* Modifiziert nach http://manning.spindoczine.com/sbe/files/uts2/Chapter12html/Chapter12.htm
*/   
class SmallButton extends JButton {
	
	public SmallButton( Action act, ImageIcon icon, String tip ) {
		super( icon );
		setValues( act,tip );
	}
		
	public SmallButton( Action act, String tip ) {
		this( act, (ImageIcon)act.getValue(Action.SMALL_ICON), tip );
	}
	
	public SmallButton( Action act ) {
		this( act, (ImageIcon)act.getValue(Action.SMALL_ICON),
			 ( String )act.getValue( Action.SHORT_DESCRIPTION ) );
	}
	
	public void setValues( Action act, String tip) {
		
		setMargin( new Insets(1,1,1,1) );
		setToolTipText( tip );
		addActionListener( act );
		setRequestFocusEnabled( false );
		
	}	
}

class NekjeConsole extends JFrame implements ActionListener {
	protected JTextArea textArea;
	protected Font textFont;
	protected Font font;
	protected Nekje pp;
	protected Document doc;
	
	protected String classpath = "."; 
	protected String warning = "";
	public NekjeConsole( Nekje parent ) {
	
		super("Nekje Console");
		pp = parent;
		
		getContentPane().setLayout( new BorderLayout() );
		
		textArea = new JTextArea();
		textArea.setLineWrap( false );
		textFont = new Font("Monospaced", Font.PLAIN, 12 );
		textArea.setFont( textFont );
		textArea.setEditable(false);
		doc = textArea.getDocument();
		
		JScrollPane scrollPane = new JScrollPane( textArea );
		
		JPanel buttonPanel = new JPanel();
		
		JButton closeButton = new JButton("Schlieen");
		closeButton.setActionCommand("Close");
		closeButton.addActionListener( this );
		
		JButton printButton = new JButton("Drucken");
		printButton.setActionCommand("Print");
		printButton.addActionListener( this);
		
		buttonPanel.add(closeButton );
		buttonPanel.add(printButton );
		
		getContentPane().add( scrollPane, BorderLayout.CENTER);
		getContentPane().add( buttonPanel, BorderLayout.SOUTH);
		
		addWindowListener(
			new WindowAdapter() {
				public void windowClosing(WindowEvent e) {
					setVisible(false);
				}
			}
		);
	
		setSize(600,400);
		setLocation(pp.getX()+50,pp.getY()+100);
		setVisible(false);
	}
	
	public void actionPerformed( ActionEvent e ) {
		
		String command = e.getActionCommand();
		
		if( command.equals("Print") ) {		
			this.printConsoleText();
		}
	       	else if( command.equals("Close") ) {
			this.setVisible(false);        	
	       	}
	}
    	
	public void processSystemCommand( final String command ) {
		Thread runner = new Thread() {
			public void run() {
				try {
					Process p = Runtime.getRuntime().exec(
					
					"c:\\windows\\command.com " + command);
				}
				catch( IOException e ) {
				}
			}
		};
			
		runner.start();
	}
	public void processCommand( final String command ) {
		Thread runner = new Thread() {
			public void run() {
				textArea.setText("");
				setVisible(true);
				textArea.append( warning );
				textArea.append("[BEFEHL:] " + command + "\n\n");
				
				String text="";
	
				try {
					Process p = Runtime.getRuntime().exec(command);
							
					BufferedReader in1 = new BufferedReader( 
						new InputStreamReader( p.getErrorStream() ) ); 
							
					
					// Ausgabe von STDERR
					boolean err = false;
					textArea.append("[FEHLER:] ");
					
					while( ( text = in1.readLine() ) != null ) {
						if( err == false )						
							err = true;
						textArea.append( "\n\t" + text );
					}
				
					if( err == false ) {
						textArea.append("Keine Fehler gefunden.\n");
					}

					in1.close();
							
							
					// Ausgabe von STDOUT			
					textArea.append("\n[STDOUT:]\n");
					
					BufferedReader in2 = new BufferedReader( 
						new InputStreamReader( p.getInputStream() ) ); 
					
					int i = 0;
					while( ( text = in2.readLine() ) != null && i < 4) {
						textArea.append( "\t" + text +"\n" );

					}
					in2.close();
					
					textArea.append( "\n[EXIT-WERT:] " + p.exitValue() + "\n");
				}
				catch( IOException ex ) {
					textArea.append( ex.toString() );
				}
			}
		};
			
		runner.start();
	}

	synchronized void printConsoleText() {
		
		// Get a PrinterJob
   		final PrinterJob job = PrinterJob.getPrinterJob();
   		PageFormat pf = job.defaultPage();
    		
 		Book bk = new Book();
   		DocumentPainter dp = new DocumentPainter( doc, pf, false );
   		int pnum = dp.calculatePageCount();
   		bk.append( dp, pf, pnum );
   		
   		// Pass the book to the PrinterJob
   		job.setPageable( bk );
    		
 		// Put up the dialog box
 		if (job.printDialog()) {
 	     	Thread runner = new Thread() {
    	   		public void run() {
    		
       				// Print the job if the user didn't cancel printing
    				try { job.print(); }
    				catch (Exception exc) { /* Handle Exception */ }
    			}
           	};
           	runner.start();	
    	}
 	}
	
	     	    	
	public void compileJavaApp( String jDir, String oDir, String jFile ) {	
		
		String cp = "";
		String outp = "";
		String pcp = pp.getProperties().getProperty("classpath");
		String pathSep = pp.getProperties().getProperty("pathSeparator");
		String outputPath = pp.getProperties().getProperty("outputDirectory");
		
		warning = "";
		
		if( pathSep == null ) {
			// Setze Default
			pathSep = ";";
		}
		
		if( pcp != null && pcp.length() > 0 )
			cp = pcp + pathSep + jDir;
		else	
			cp = jDir;
			
		if( outputPath != null && outputPath.length() > 0 )
			outp = outputPath;
		else
			outp = oDir;
					
		processCommand( pp.getProperties().getProperty("javacExe") + 
					" -classpath " + cp + " -d " + outp + " " + jDir+jFile );
	}
	
	/**
	* Ruft den Java-Interpreter auf
	*/
	public void runJavaApp( String clDir, String clFile, String clParameter, 
				boolean systemFlag ) {	
		 
		// Pruefe, ob Standardsuffix .java oder .class vorhanden.
		// Bei Standardsuffix schneide Suffix ab, sonst lasse den
		// Dateinamen unveraendert und riskiere Fehlermeldung von
		// Java
		String str = clFile;
		String workingDir = clDir;
		String cp = clDir;
		String pcp = pp.getProperties().getProperty("classpath");
		String pathSep = pp.getProperties().getProperty("pathSeparator");
		String outputPath = pp.getProperties().getProperty("outputDirectory");
		warning = "Hinweis: Umfangreiche DOS-Konsolenausgaben fuehren unter Windows 95/98\nleider zum Haengen von Prozessen (siehe hierzu Nekje-Hilfe)\n\n";
		
		if( pathSep == null ) {
			// Setzte Default
			pathSep = ";";
		}
		
		// Muessen beim classpath Ergaenzungen vorgenommen werden?
		if( pcp != null && pcp.length() > 0 )
				cp = pcp + pathSep + clDir;
		else
			cp = clDir;
		
		if( outputPath != null && 
			outputPath.equals( clDir ) == false ){
			cp += pathSep+outputPath;
		}
		
		// Wenn Applikation, rufe JVM auf.
		if( str.endsWith(".java") || str.endsWith(".class") ) {
		
			str = str.substring(0, str.lastIndexOf('.') );

			// Anhaengen von evtl. Kommandozeilen-Parametern
			if( clParameter != null )
				str += " " + clParameter;
			
			String cmd = pp.getProperties().getProperty("javaExe") + 
						" -classpath " + cp + " " + str;	
			if( systemFlag == false ) {	
				// Uebergebe String an Nekje-Shell
				processCommand( cmd );
			}		
			else {
				// Uebergeb an System-Shell
				processSystemCommand( cmd );
			}
		}
		// Wenn HTML mit Applet, starte Appletviewer
		else if( str.endsWith(".htm") || str.endsWith(".html") ) {
			// Anhaengen von evtl. Kommandozeilen-Parametern
			if( clParameter != null )
				str += " " + clParameter;

			System.setProperty("user.dir", workingDir);
			processCommand( pp.getProperties().getProperty("appletviewerExe") 
				+ " file:///" + workingDir + str );

		}
	}

	public void showDocument( final Document doc ) {
	
		final Element elem = doc.getDefaultRootElement();
		
		final int elemCount = elem.getElementCount();
		
		Thread runner = new Thread() {
			public void run() {
				
				int start, end;
				textArea.setText("");
				setVisible(true);
				for(int i = 0; i < elemCount; i++ ) {
					Element cElem = elem.getElement(i);
					start = cElem.getStartOffset();
					end = cElem.getEndOffset();
					
					textArea.append("\nElement (" + start +" " + end + ") " + i + ": ");
					try {
						int len = end-start;
						String cText = doc.getText( start,len);
						textArea.append(cText);
					}
					catch(BadLocationException ex) {}
				}
				
				if( pp.getProperties() != null ) {
					String str = pp.getProperties().getProperty("javaExe");
					
					textArea.append("Property javaExe = " + str + "\n");
				}
				else {
					textArea.append("Sorry. Keine Properties gefunden.\n");
				}
				
			}
		};
			
		runner.start();
	}
	
	public void setErrorColor() {
		textArea.setForeground( Color.red);
	}
	
	public void setStandardColor() {
		textArea.setForeground( Color.black );
	}
	
	public void appendText( String text ) {
		textArea.append( text );
	}
	
}


class DocumentPainter extends JComponent implements Printable {
    		
    	protected Document doc;
    	protected PageFormat pFormat;
    	protected ArrayList pages;
    	protected Font font;
    	protected boolean rowNumbers;	
    	
    	private int currentPage;
    	private Dimension mPreferredSize;
    		
    	public DocumentPainter( Document d, PageFormat pf, boolean rowNumbers ) {
    		super();
    		doc = d;
    		pFormat = pf;
    		this.rowNumbers = rowNumbers;
    		
       		font = new Font("Monospaced", Font.PLAIN, 10);
   		
    		pages = repaginate(pf);
    		
    		
    	}
      	
      	public Dimension getPreferredSize() { return mPreferredSize; }
  	
    	public int calculatePageCount() {
    		
    		if( pages == null )
    			pages = repaginate( pFormat );
    		return pages.size();
    	}
    	
    		
    	public ArrayList repaginate( PageFormat pf ) {
    		
    		int maxh = (int)pf.getImageableHeight();
    		int lineh = font.getSize();
    		
    		Element root = doc.getDefaultRootElement();
    		int elemNum = root.getElementCount();
    		
    		
    		ArrayList pgs = new ArrayList();
    		
    		int pageh = 0;
    		ArrayList page = new ArrayList();
    		
    		for( int i=0; i < elemNum; i++ ) {
    		
    			Element line = root.getElement(i);
    			
    			try {
    			 	CodeLine cl = new CodeLine( 
    			 		doc.getText(line.getStartOffset(), 
    						line.getEndOffset()-line.getStartOffset() ), i+1 );
    					
    				if( pageh + lineh > maxh ) {
    			
    					pgs.add( page );
    					page = new ArrayList();
    					pageh= 0;
    				}
    				page.add( cl );
    				pageh += lineh;    						
    			}
    			catch( BadLocationException e) {
    				break;		
    			}
   			}
    		
    		// Fuegt die letzte Seite ein
    		pgs.add(page);
    		
    		return( pgs ); 
    	}
    	
  	public int print(Graphics g, PageFormat pf, int pageIndex) 
  						throws PrinterException {
         if( pFormat != pf ) {
			pFormat = pf;
			pages = repaginate( pf );
		}
        	
	if( pageIndex >= pages.size() )
			return Printable.NO_SUCH_PAGE;
			    
	currentPage = pageIndex;
		 		
      	Graphics2D  g2 = (Graphics2D) g;
    	g2.translate(pf.getImageableX(), pf.getImageableY());		
    	mPreferredSize = new Dimension((int)pf.getImageableWidth(),
        	(int)pf.getImageableHeight());				
		System.out.println("Seite " + pageIndex);
		paintComponent( g2 );
		
		return Printable.PAGE_EXISTS;
      	}
      		

	public void paintComponent( Graphics g ) {
		
		Graphics2D g2 = (Graphics2D) g;
		FontMetrics fm = g.getFontMetrics();
		
		int numwidth = fm.stringWidth("00000: ");
		
        	int y = font.getSize();
	    	// Make the background white.
    		java.awt.geom.Rectangle2D r = new java.awt.geom.Rectangle2D.Float(0, 0,
      		mPreferredSize.width, mPreferredSize.height);
    		g2.setPaint(Color.white);
    		g2.fill(r);
        	
        	ArrayList page = (ArrayList)pages.get(currentPage);
        	Iterator it = page.iterator();
        	
        	g2.setPaint(Color.black);
      		g2.setFont( font );

        	while( it.hasNext() ) {

	        	int x = 0;

        		// An dieser Stelle erfolgt eine Neuformatierung des Strings.
        		CodeLine cl = (CodeLine)it.next();
        		String tmpstr = cl.getLine();
        		
        		StringBuffer temp = new StringBuffer();
        		int i = 0;
        		int pos = 0;
        		for( ; i < tmpstr.length(); i++) {
        		
        			// Ersetze Tabulatoren durch Leerzeichen
        			if( tmpstr.charAt(i) == '\t' ) {
        				if( i > pos )
        					temp.append( tmpstr.substring(pos,i));
        				temp.append("    ");
        				pos = i+1;
        			}	
        		}
				if( i > pos )
					temp.append( tmpstr.substring(pos,i) );
					
			if( rowNumbers == true ) {
				g2.drawString( cl.getNumLine() + ":", x, y);
				x = numwidth+ 10;					
			}
	
        		g2.drawString( temp.toString(), x, y);
				y += font.getSize();
       		}	
	}
	
	class CodeLine {
	
		private String line;
		private int numLine;
		
		public CodeLine( String line, int numLine ) {
			this.line = line;
			this.numLine = numLine;
		}
		
		public String getLine() { return line; }
		public int getNumLine() { return numLine; }
		
	}
	
}
																																																																																																																	   				    	    	    		    	   		   																		     						      						 	     				      												    				    											 		 			 																																																																																																																								        	        	      		      																																																																																     																																																													    	    			    			 																			 	     				      														    				    											 		 			 																																																																																																																						        	        	      		      																																																																																     																																																													    	    			    			 																			 	     				      														    				    											 		 			 																																																																																																																					