package javakurs.io;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

/**
*	Die Klasse stellt ein Eingabefenster bereit, mit dem
*	Daten in ein Programm eingelesen werden k&ouml;nnen.
*
*	<p>Wichtig: Falls in einer Applikation nur die Klasse Eingabe
*	Verwendung findet, so muss die Anwendung mit einem
*	Aufruf der Methode System.exit() abgeschlossen werden, damit
*	der Programmprozess beendet wird.
* 
*
*       <p>Beispiel f&uuml;r den Gebrauch der Klasse:
* 	<pre>
*	import javakurs.io.*;
* 	public class EingabeDemo {
*		public static void main( String[] args ) {
*			Eingabe.read( "Geben Sie zwei Zahlwerte ein:" );
*			double wert1 = Eingabe.getDouble(0);
*			double wert2 = Eingabe.getDouble(1);	
*			System.exit(0);	
*		}
*	}
* </pre>
*
*	<p>Das Eingabefenster hat f&uuml;r das Beispiel EingabeDemo das folgende Aussehen:<br><br>
*	<img src="Eingabe.png">
*
* 	@author Hubertus Espel
* 	@version 00-09-30
*
*/
public final class Eingabe {

	private static String eingabeString = "";
	private static final SimpleInput in = new SimpleInput();
	
	private Eingabe(){
	}
	
	/**
	*	Setzt die Werte, die als Begrenzer zwischen mehreren Eingabewerten
	*	erkannt werden. Als Standardwerte sind das Leerzeichen und der Zeilenumbruch
	*	definiert (" \n").
	*
	*	@param begrenzer Referenz auf ein Stringobjekt mit den Begrenzerwerten.
	*
	*/
	public static void setDelimiter( String begrenzer ) {
		in.setDelimiter( begrenzer );
	}
	
	/**
	*	Bestimmt, ob das Eingabefenster nach erfolgter Eingabe
	*	geschlossen wird.
	*
	*	@param wert true: das Fenster wird geschlossen, false: das Fenster bleibt sichtbar.
	*/
	public static void setPermanent( boolean wert ) {
		in.setPermanent( wert );
	}
	
	/**
	*	Das Eingabefenster wird aktiviert. Wenn es zuvor
	*	nicht sichtbar war, so wird es jetzt ge&ouml;ffnet.
	*	Der Programmfluss wird bis zum Abschluss der Eingabe
	*	unterbrochen.
	*
	*	@return Referenz auf ein Stringobjekt, welches die gesamte Eingabe enth&auml;lt.
	*/
	public static String read() {
		return( in.read() );
	}
	
	/**
	*	Das Eingabefenster wird aktiviert. Wenn es zuvor nicht
	*	sichtbar war, so wird es jetzt ge&ouml;ffnet.
	*	Der Programmfluss wird bis zum Abschluss der Eingabe
	*	unterbrochen.
	*
	*	@param prompt Eingabeaufforderung an den Benutzer
	*	@return Referenz auf ein Stringobjekt, welches die gesamte Eingabe enth&auml;lt.
	*/
	public static String read( String prompt ) {
		return( in.read( prompt ) );	
	}
	
	/**
        *      Identisch mit read().
	*/
	public static String readln() {
		return( in.readln() );
	}
	
	/**
        *       Identisch mit read( String prompt).
	*/
	public static String readln( String prompt ) {
		return( in.readln( prompt ) );	
	}
	
	/**
	*	Liefert eine Referenz auf ein Stringobjekt, welches den gesamten Eingabestring enth&auml;lt.
	*/
	public static String getText() {
		return( in.getText() );	
	}
	
	/**
	*	Liefert eine Referenz auf ein Stringobjekt, welches den Eingabewert an der bestimmten
	*	Position enth&auml;lt.
	*
	*	@param pos Position des Eingabewertes innerhalb der Gesamteingabe.
	*/
	public static String getText( int pos ) {
		return in.getText( pos );
	}
	
	/**
        *       Liefert den Eingabewert an der bestimmten Position als Wert vom Datentyp char zur&uuml;ck.
	*
	*	@param pos Position des Eingabewertes innerhalb der Gesamteingabe.
	*
	*	@return Wert an der bestimmten Position als char. Besteht der Wert aus mehreren Textzeichen, so wird
        *       nur das erste Textzeichen zur&uuml;ckgegeben.
	*/
	public static char getChar( int pos ) {
		return in.getChar( pos );
	}

	/**
        *       Liefert den Eingabewert an der bestimmten Position als Wert vom Datentyp int zur&uuml;ck.
	*
	*	@param pos Position des Eingabewertes innerhalb der Gesamteingabe.
	*
	*	@return Wert n der bestimmten Position als int. Handelt es sich nicht um einen Zahlwert, so wird
	*	0 zur&uuml;ckgegeben
	*/
	public static int getInt( int pos ) {
		return in.getInt( pos );
	}
	
	/**
        *       Liefert den Eingabewert an der bestimmten Position als Wert vom Datentyp long zur&uuml;ck.
	*
	*	@param pos Position des Eingabewertes innerhalb der Gesamteingabe.
	*
	*	@return Wert n der bestimmten Position als long. Handelt es sich nicht um einen Zahlwert, so wird
	*	0L zur&uuml;ckgegeben
	*/
	public static long getLong( int pos ) {
		return in.getLong( pos );
	}


	/**
        *       Liefert den Eingabewert an der bestimmten Position als Wert vom Datentyp double zur&uuml;ck.
	*
	*	@param pos Position des Eingabewertes innerhalb der Gesamteingabe.
	*
	*	@return Wert n der bestimmten Position als double. Handelt es sich nicht um einen Zahlwert, so wird
	*	0.0 zur&uuml;ckgegeben
	*/
	public static double getDouble( int pos ) {
		return in.getDouble( pos );
	}

	
	/**
        *       Liefert den Eingabewert an der bestimmten Position als Wert vom Datentyp float zur&uuml;ck.
	*
	*	@param pos Position des Eingabewertes innerhalb der Gesamteingabe.
	*
	*	@return Wert n der bestimmten Position als float. Handelt es sich nicht um einen Zahlwert, so wird
	*	0.0F zur&uuml;ckgegeben
	*/
	public static float getFloat( int pos ) {
		return in.getFloat( pos );
	}
	

	/**
	*	Gibt nach erfolgter Eingabe die Anzahl der Eingabewert zur&uuml;ck.
	*
	*	@return Anzahl der Eingabewerte in Abh&auml;ngigkeit der Begrenzer.
	*/
	public static int size( ) {
		return in.vectorSize();
	}
}

class SimpleInput extends Frame implements ActionListener{
	private TextArea tf;
	private Button ok;
	private Label lb;
	private boolean ready = false;
	private boolean permanent = false;
	private String inString = "";
	private Vector inVector;
	private String delim = " \n\t";
		
	public SimpleInput() {
		setTitle("Objekt der Klasse javakurs.io.SimpleInput");
		tf = new TextArea("",1,20, TextArea.SCROLLBARS_NONE);
		ok = new Button("OK");
		lb = new Label("Eingabe: ");
		ok.addActionListener( this );
		
		add( lb, BorderLayout.NORTH );	
		add( tf, BorderLayout.CENTER );
		add( ok, BorderLayout.SOUTH );
		
		this.addWindowListener(
			new WindowAdapter() {
				// Bei Programmexit-Event beende Programm
				public void windowClosing(WindowEvent e) {
					System.exit(0);
				}
			}
		);		
		Dimension dim = getToolkit().getScreenSize();
		setLocation(0,0);
		setSize( dim.width/2, dim.height/4 );
		setLocation(0,0);
	}	
		
	public void actionPerformed( ActionEvent e ) {
		inString = tf.getText();
		tf.requestFocus();	
		ready = true;
	}	

	public boolean isReady() {
		return ready;	
	}
	
	public void setDelimiter( String delimiter ) {
		delim = delimiter;	
	}
	
	public void setReady( boolean value ) {
		ready = value;
	}
	
	public void setPermanent( boolean value ) {
		permanent = value;
	}
	
	public String getText() {
		return inString;
	}	

	public String getText( int pos ) {
		if( pos < inVector.size() )
			return (String)inVector.elementAt(pos);		
		else
			return("");	
	}
	
	public char getChar( int pos ) {
		char temp = ' ';
		
		if( pos < inVector.size() ){
				temp = ((String)inVector.elementAt(pos)).charAt(0);
		}
		return temp;	
	}
	

	public int getInt( int pos ) {
		int temp = 0;
		
		if( pos < inVector.size() ){
			try {				
				temp = Integer.parseInt( (String)inVector.elementAt(pos) );
			}
			catch( NumberFormatException e ) {
			}	
		}

		return temp;	
	}

	public long getLong( int pos ) {
		long temp = 0L;
		
		if( pos < inVector.size() ){
			try {				
				temp = Long.parseLong( (String)inVector.elementAt(pos) );
			}
			catch( NumberFormatException e ) {
			}	
		}

		return temp;	
	}

	public double getDouble( int pos ) {
		double temp = 0;
		
		if( pos < inVector.size() ){
			try {				
				temp = Double.valueOf( (String)inVector.elementAt(pos) ).doubleValue();
			}
			catch( NumberFormatException e ) {
			}	
		}

		return temp;	
	}

	public float getFloat( int pos ) {
		float temp = 0F;
		
		if( pos < inVector.size() ){
			try {				
				temp = Double.valueOf( (String)inVector.elementAt(pos) ).floatValue();
			}
			catch( NumberFormatException e ) {
			}	
		}

		return temp;	
	}	

	public String read( String prompt ) {
		lb.setText( prompt );
		return( this.read() );
	}
	
	public String read() {
		
		String temp = this.readln();
		
		StringTokenizer tk = new StringTokenizer( temp, delim );
		
		if( inVector == null )
			inVector = new Vector();
		else
			inVector.clear();

		while( tk.hasMoreTokens() ) {
			inVector.addElement( tk.nextToken() );	
		}					

		return( temp );
	}
	
	public String readln( String prompt ) {
		lb.setText( prompt );
		return this.readln();
	}
	
	public String readln() {
		try {
			Thread runner = new Thread() {
				public void run() {
					tf.setText("");
					setReady( false );
					
					if( isVisible() == false )
						setVisible( true );

					while( isReady() == false ) {
						try {
							sleep(20);
						}
						catch( InterruptedException e ){
							break;
						}					
					}
					
					if( permanent == false ){
						setVisible( false );
					}	
				}					
			};
		
			runner.start();
			runner.join();
			runner.interrupt();
		}
		catch( InterruptedException e ) {
		}

		return tf.getText();		
	}	
	
	public int vectorSize() {
		int n = 0;
		if( inVector != null )
			n = inVector.size();
		return n;			
	}
}
																																																																																																																																																																																																																																																																																																																																											
